import {DraggableAboveNodes} from '@plate/components/draggable';
import {ElementApi, NodeApi, NodeEntry, PathApi, TElement, TextApi} from '@udecode/plate';
import {AlignPlugin} from '@udecode/plate-alignment/react';
import {
    BoldPlugin,
    CodePlugin,
    ItalicPlugin,
    StrikethroughPlugin,
    SubscriptPlugin,
    SuperscriptPlugin,
    UnderlinePlugin
} from '@udecode/plate-basic-marks/react';
import {ExitBreakPlugin, SoftBreakPlugin} from '@udecode/plate-break/react';
import {CsvPlugin} from '@udecode/plate-csv';
import {DndPlugin} from '@udecode/plate-dnd';
import {DocxPlugin} from '@udecode/plate-docx';
import {
    FontBackgroundColorPlugin,
    FontColorPlugin,
    FontFamilyPlugin,
    FontSizePlugin
} from '@udecode/plate-font/react';
import {HEADING_KEYS} from '@udecode/plate-heading';
import {HeadingPlugin} from '@udecode/plate-heading/react';
import {HighlightPlugin} from '@udecode/plate-highlight/react';
import {IndentPlugin} from '@udecode/plate-indent/react';
import {LineHeightPlugin} from '@udecode/plate-line-height/react';
import {
    BaseListItemContentPlugin,
    BaseListItemPlugin,
    BaseListPlugin,
    getListTypes,
    toggleList,
    unwrapList
} from '@udecode/plate-list';
import {ListPlugin} from '@udecode/plate-list/react';
import {ImagePlugin} from '@udecode/plate-media/react';
import {SelectOnBackspacePlugin} from '@udecode/plate-select';
import {TablePlugin} from '@udecode/plate-table/react';
import {AnyPlatePlugin, ParagraphPlugin} from '@udecode/plate/react';
import {TypedEmitter} from 'tiny-typed-emitter';
import {linkPlugin} from '@plate/plugins/link-plugin';
import {EmojiPlugin} from '@udecode/plate-emoji/react';
import emojiMartData from '@emoji-mart/data';
import {BlockquotePlugin} from '@udecode/plate-block-quote/react';
import {NodeIdPlugin} from '@udecode/plate-node-id';

export const PARAGRAPH_KEYS = [ParagraphPlugin.key,
    HEADING_KEYS.h1, HEADING_KEYS.h2, HEADING_KEYS.h3, HEADING_KEYS.h4, HEADING_KEYS.h5, HEADING_KEYS.h6];

export const createPlatePlugins = (emitter?: TypedEmitter): AnyPlatePlugin[] => [
    // Node Id
    NodeIdPlugin.configure({
        options: {normalizeInitialValue: true},
    }),

    // Nodes
    ParagraphPlugin.overrideEditor(({editor, tf: {normalizeNode}}) => {
        return {
            transforms: {
                // wrap consecutive text nodes in paragraph
                normalizeNode([node, path]) {
                    if (NodeApi.isEditor(node)) {
                        let wasNextText = false;
                        for (const [child, childPath] of Array.from(NodeApi.children(node, path, {reverse: true}))) {
                            if (TextApi.isText(child)) {
                                if (wasNextText) {
                                    editor.tf.moveNodes({
                                        at: childPath,
                                        to: PathApi.firstChild(PathApi.next(childPath))
                                    });
                                } else {
                                    editor.tf.wrapNodes({
                                        type: ParagraphPlugin.key,
                                        children: [child]
                                    }, {at: childPath});
                                }
                                wasNextText = true;
                            } else wasNextText = false;
                        }
                        // ensure editor remains non-empty
                        if (node.children.length === 0) {
                            editor.tf.insertNode({
                                type: ParagraphPlugin.key,
                                children: [{text: ''}]
                            });
                        }
                    }
                    normalizeNode([node, path]);
                },
            },
        };
    }),
    HeadingPlugin,
    BlockquotePlugin,
    TablePlugin,

    // Paragraph Style
    AlignPlugin.configure({inject: {targetPlugins: PARAGRAPH_KEYS}}),
    // ListPlugin must be before IndentPlugin for tab and shift+tab to work
    ListPlugin.configure({options: {enableResetOnShiftTab: true, validLiChildrenTypes: Object.values(HEADING_KEYS)}})
        .extendEditorTransforms(({editor}) => ({
            toggle: {
                list: ({type}) =>
                    editor.tf.withoutNormalizing(() => {
                        /* Potix: Fix toolbar action after grid cell selection */
                        const {selectedCells} = editor.getOptions(TablePlugin);
                        if (selectedCells && selectedCells.length > 0) {
                            const shouldUnwrapList = selectedCells.every(cell =>
                                cell.children.length === 1 && cell.children[0].type === type
                            );
                            if (shouldUnwrapList) {
                                selectedCells.forEach(cell => {
                                    const cellPath = editor.api.findPath(cell);
                                    if (cellPath) unwrapList(editor, {at: cellPath});
                                });
                                return;
                            }
                            const {validLiChildrenTypes} = editor.getOptions(BaseListPlugin);
                            selectedCells.forEach(cell => {
                                const cellPath = editor.api.findPath(cell);
                                if (cellPath) {
                                    const commonEntry: NodeEntry = [cell, cellPath];
                                    const rootPathLength = commonEntry[1].length;
                                    const _nodes = editor.api.nodes<TElement>({
                                        at: commonEntry[1],
                                        mode: 'all',
                                    });
                                    const nodes = Array.from(_nodes).filter(
                                        ([, path]) => path.length === rootPathLength + 1
                                    );

                                    nodes.forEach((n) => {
                                        if (getListTypes(editor).includes(n[0].type)) {
                                            editor.tf.setNodes(
                                                {type},
                                                {
                                                    at: n[1],
                                                    mode: 'all',
                                                    match: (_n) =>
                                                        ElementApi.isElement(_n) &&
                                                        getListTypes(editor).includes(_n.type),
                                                }
                                            );
                                        } else {
                                            if (!validLiChildrenTypes?.includes(n[0].type)) {
                                                editor.tf.setNodes(
                                                    {type: editor.getType(BaseListItemContentPlugin)},
                                                    {at: n[1]}
                                                );
                                            }

                                            const listItem = {
                                                children: [],
                                                type: editor.getType(BaseListItemPlugin),
                                            };
                                            editor.tf.wrapNodes<TElement>(listItem, {
                                                at: n[1],
                                            });

                                            const list = {children: [], type};
                                            editor.tf.wrapNodes<TElement>(list, {at: n[1]});
                                        }
                                    });

                                }
                            });
                        } else toggleList(editor, {type});
                    })
            }
        })),
    IndentPlugin.configure({inject: {targetPlugins: PARAGRAPH_KEYS}}),
    LineHeightPlugin.configure({
        inject: {
            nodeProps: {
                defaultNodeValue: 1.5,
                validNodeValues: [1, 1.15, 1.5, 2, 2.5, 3],
            },
            targetPlugins: PARAGRAPH_KEYS,
            targetPluginToInject: ({editor, plugin}) => ({
                parsers: {
                    html: {
                        deserializer: {
                            parse: ({element}) => {
                                if (element.style.lineHeight) {
                                    const lineHeight = Number(element.style.lineHeight);
                                    if (!isNaN(lineHeight)) return {[editor.getType(plugin)]: lineHeight};
                                }
                            },
                        },
                    },
                },
            }),
        }
    }),

    // Marks
    BoldPlugin,
    ItalicPlugin,
    StrikethroughPlugin,
    SubscriptPlugin,
    SuperscriptPlugin,
    UnderlinePlugin,
    CodePlugin,
    FontBackgroundColorPlugin,
    FontColorPlugin,
    FontFamilyPlugin,
    FontSizePlugin,
    HighlightPlugin,
    SoftBreakPlugin,

    linkPlugin,
    EmojiPlugin.configure({
        options: {
            data: emojiMartData as emojiMartData.EmojiMartData
        }
    }),

    // Deserialization
    CsvPlugin,
    DocxPlugin,

    // Media
    ImagePlugin,
    SelectOnBackspacePlugin.configure({
        options: {
            query: {
                allow: [ImagePlugin.key],
            },
        },
    }),

    // Drag and Drop
    DndPlugin.configure({
        options: {enableScroller: true},
        render: {aboveNodes: DraggableAboveNodes},
    }),
    ExitBreakPlugin.configure({
        options: {
            rules: [
                {
                    hotkey: 'mod+enter',
                },
                {
                    before: true,
                    hotkey: 'mod+shift+enter',
                },
            ],
        },
    }),
] as AnyPlatePlugin[];
