class ClipboardHelper {
    static CLIPBOARD_ACTION_EVENT = 'onClipboardAction';
    static targetComponentUuid; //if null, sending to the current desktop

    /**
     * Writes text to the clipboard
     * @param {string} text - The text to write to the clipboard
     */
    static writeText(text) {
        if (navigator.clipboard) {
            navigator.clipboard.writeText(text)
                .then(() => {
                    this.fireEventToServer({ action: 'WRITE' });
                })
                .catch(error => {
                    this.fireEventToServer({
                        action: 'WRITE',
                        error: this.extractError(error),
                    });
                });
        } else {
            this.fireEventToServer({
                action: 'WRITE',
                error: {message:'Clipboard API not supported by this browser.'}
            });
        }
    }

    /**
     * Reads text from the clipboard
     */
    static readText() {
        if (navigator.clipboard) {
            navigator.clipboard.readText()
                .then(text => {
                    this.fireEventToServer({
                        action: 'READ',
                        text: text
                    });
                })
                .catch(error => {
                    this.fireEventToServer({
                        action: 'READ',
                        error: this.extractError(error)
                    });
                });
        } else {
            this.fireEventToServer({
                action: 'READ',
                error: {message: 'Clipboard API not supported by this browser.'}
            });
        }
    }

    /**
     * Reads image data from the clipboard
     * Uses the modern clipboard.read() API to access ClipboardItem objects
     */
    static readImage() {
        // Check if the modern clipboard API is supported
        if (!navigator.clipboard || !navigator.clipboard.read) {
            this.fireEventToServer({
                action: 'READ_IMAGE',
                error: {message: 'Clipboard read() API not supported by this browser. Requires Chrome 88+, Firefox 127+, or Edge 88+.'}
            });
            return;
        }

        navigator.clipboard.read()
            .then(clipboardItems => {
                // Look for image data in clipboard items
                for (const item of clipboardItems) {
                    // Check each type in the clipboard item
                    for (const type of item.types) {
                        if (type.startsWith('image/')) {
                            // Found an image, process it
                            return item.getType(type).then(blob => {
                                return this.processImageBlob(blob, type);
                            });
                        }
                    }
                }
                
                // No image found
                this.fireEventToServer({
                    action: 'READ_IMAGE',
                    error: {message:'No image data found in clipboard'}
                });
            })
            .catch(error => {
                console.error(error);
                this.fireEventToServer({
                    action: 'READ_IMAGE',
                    error: this.extractError(error)
                });
            });
    }

    /**
     * Processes an image blob and converts it to base64 for transfer to server
     * @param {Blob} blob - The image blob from clipboard
     * @param {string} mimeType - The MIME type of the image
     */
    static processImageBlob(blob, mimeType) {
        this.blobToBase64(blob)
            .then(base64Data => {
                // Create image to get dimensions
                const img = new Image();
                img.onload = () => {
                    this.fireEventToServer({
                        action: 'READ_IMAGE',
                        mimeType: mimeType,
                        imageData: base64Data,
                        size: blob.size,
                        width: img.width,
                        height: img.height
                    });
                };
                
                img.onerror = () => {
                    // If we can't load the image for dimensions, send without them
                    this.fireEventToServer({
                        action: 'READ_IMAGE',
                        mimeType: mimeType,
                        imageData: base64Data,
                        size: blob.size,
                        width: 0,
                        height: 0
                    });
                };
                
                // Create object URL and load image
                const objectUrl = URL.createObjectURL(blob);
                img.src = objectUrl;
                
                // Clean up object URL after a short delay
                setTimeout(() => {
                    URL.revokeObjectURL(objectUrl);
                }, 1000);
            })
            .catch(error => {
                this.fireEventToServer({
                    action: 'READ_IMAGE',
                    error: this.extractError(error)
                });
            });
    }

    /**
     * Converts a blob to base64 string
     * @param {Blob} blob - The blob to convert
     * @returns {Promise<string>} Promise that resolves to base64 string (without data: prefix)
     * @private
     */
    static blobToBase64(blob) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => {
                // Remove the data:image/xxx;base64, prefix
                const base64 = reader.result.split(',')[1];
                resolve(base64);
            };
            reader.onerror = reject;
            reader.readAsDataURL(blob);
        });
    }

    /**
     * Fires an event to the server with the given data
     * @param {Object} data - The data to send to the server
     * @private
     */
    static fireEventToServer(data) {
        zAu.send(new zk.Event(this.getEventTarget(), ClipboardHelper.CLIPBOARD_ACTION_EVENT, data));
    }

    static extractError(error){
        return {code: error.code, message: error.message};
    }

    /**
     * Sets the target component UUID for the next clipboard operation.
     * This enables targeted event delivery - the clipboard event will be sent to the specific
     * component instead of broadcasting to the desktop event queue.
     *
     * This method is called from Java via Clients.evalJavaScript() when using the
     * readTextTo(), readImageTo(), or writeTextTo() APIs.
     *
     * @param {string} uuid - The UUID of the target ZK component. If null or undefined,
     *                         events will be broadcast to the desktop.
     * @private
     */
    static setTargetComponentUuid(uuid){
        ClipboardHelper.targetComponentUuid = uuid;
    }

    /**
     * Gets the event target for clipboard operations.
     * Returns the widget matching the target component UUID if set, otherwise returns
     * the current desktop for broadcast delivery.
     *
     * This method is used internally by fireEventToServer() to determine where
     * clipboard events should be sent:
     * - If targetComponentUuid is set: delivers event to that specific widget
     * - If targetComponentUuid is null: delivers event to desktop (broadcast)
     *
     * @returns {Object} Either the target ZK widget or zk.Desktop._dt (the desktop)
     * @private
     */
    static getEventTarget(){
        let widget = zk.Widget.$('#' +  ClipboardHelper.targetComponentUuid);
        if (widget) {
            return widget;
        }else{
            return zk.Desktop._dt;
        }
    }

}